// Animações e interatividade do painel
document.addEventListener('DOMContentLoaded', function() {
    // Preencher campos automaticamente se vier via URL
    const urlParams = new URLSearchParams(window.location.search);
    if (urlParams.get('action') === 'login') {
        const loginUser = urlParams.get('login_user');
        const loginPassword = urlParams.get('login_password');
        
        if (loginUser) {
            const userInput = document.querySelector('input[name="login_user"]');
            if (userInput) userInput.value = loginUser;
        }
        
        if (loginPassword) {
            const passwordInput = document.querySelector('input[name="login_password"]');
            if (passwordInput) passwordInput.value = loginPassword;
        }
    }
    
    // Animação de entrada dos elementos
    const panel = document.querySelector('.panel');
    panel.style.opacity = '0';
    panel.style.transform = 'translateY(20px)';
    
    setTimeout(() => {
        panel.style.transition = 'all 0.6s ease-out';
        panel.style.opacity = '1';
        panel.style.transform = 'translateY(0)';
    }, 100);

    // Efeito de hover nos botões
    const buttons = document.querySelectorAll('.btn');
    buttons.forEach(button => {
        button.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-2px) scale(1.02)';
        });
        
        button.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(0) scale(1)';
        });
    });

    // Efeito de focus nos inputs
    const inputs = document.querySelectorAll('input');
    inputs.forEach(input => {
        input.addEventListener('focus', function() {
            this.parentElement.style.transform = 'scale(1.02)';
        });
        
        input.addEventListener('blur', function() {
            this.parentElement.style.transform = 'scale(1)';
        });
    });

    // Animação do cash display
    const cashValue = document.querySelector('.cash-value');
    if (cashValue) {
        let currentValue = parseInt(cashValue.textContent);
        let targetValue = currentValue;
        let isAnimating = false;
        
        // Função para animar mudanças no cash
        function animateCashChange(newValue) {
            if (isAnimating) return; // Evitar animações múltiplas
            
            isAnimating = true;
            // Atualizar o valor atual da tela antes de animar
            currentValue = parseInt(cashValue.textContent);
            targetValue = newValue;
            const difference = targetValue - currentValue;
            const duration = Math.min(Math.max(Math.abs(difference) / 100, 500), 2000); // 500ms a 2s
            const increment = difference / (duration / 50); // 50ms por frame
            
            // logs removidos em produção
            
            const timer = setInterval(() => {
                currentValue += increment;
                // passo de animação (silenciado)
                if ((increment > 0 && currentValue >= targetValue) || 
                    (increment < 0 && currentValue <= targetValue)) {
                    currentValue = targetValue;
                    // animação concluída
                    clearInterval(timer);
                    isAnimating = false; // Liberar para novas animações
                }
                cashValue.textContent = Math.floor(currentValue);
            }, 50);
        }
        
        // Observar mudanças no DOM para detectar novos valores de cash (apenas quando não estiver animando)
        const observer = new MutationObserver(function(mutations) {
            if (isAnimating) return; // Não interferir durante animação
            
            mutations.forEach(function(mutation) {
                if (mutation.type === 'childList' || mutation.type === 'characterData') {
                    const newValue = parseInt(cashValue.textContent);
                    if (newValue !== targetValue) {
                        animateCashChange(newValue);
                    }
                }
            });
        });
        
        observer.observe(cashValue, {
            childList: true,
            characterData: true,
            subtree: true
        });
    }

    // Efeito de partículas no fundo (opcional)
    createParticles();

    // Auto-hide das mensagens de alerta
    const alerts = document.querySelectorAll('.alert');
    alerts.forEach(alert => {
        setTimeout(() => {
            alert.style.transition = 'all 0.5s ease-out';
            alert.style.opacity = '0';
            alert.style.transform = 'translateY(-10px)';
            setTimeout(() => {
                alert.remove();
            }, 500);
        }, 5000);
    });

    // Efeito de loading nos formulários
    const forms = document.querySelectorAll('form');
    forms.forEach(form => {
        if (form.id === 'registerForm') {
            // Formulário de registro com AJAX
            form.addEventListener('submit', function(e) {
                e.preventDefault();
                
                const submitBtn = this.querySelector('button[type="submit"]');
                const messageDiv = document.getElementById('registerMessage');
                const originalText = submitBtn.textContent;
                
                // Mostrar loading
                submitBtn.innerHTML = '<span class="loading">⏳</span> Registrando...';
                submitBtn.disabled = true;
                messageDiv.innerHTML = '';
                
                // Preparar dados do formulário
                const formData = new FormData(this);
                
                // Enviar via AJAX
                fetch('index.php', {
                    method: 'POST',
                    body: formData
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        messageDiv.innerHTML = `<div class="alert alert-success">${data.message}</div>`;
                        this.reset(); // Limpar formulário
                    } else {
                        messageDiv.innerHTML = `<div class="alert alert-error">${data.message}</div>`;
                    }
                })
                .catch(error => {
                    messageDiv.innerHTML = `<div class="alert alert-error">Erro de conexão: ${error.message}</div>`;
                })
                .finally(() => {
                    // Restaurar botão
                    submitBtn.textContent = originalText;
                    submitBtn.disabled = false;
                });
            });
        } else if (form.id === 'loginForm') {
            // Formulário de login com AJAX
            form.addEventListener('submit', function(e) {
                e.preventDefault();
                
                const submitBtn = this.querySelector('button[type="submit"]');
                const messageDiv = document.getElementById('loginMessage');
                const originalText = submitBtn.textContent;
                
                // Mostrar loading
                submitBtn.innerHTML = '<span class="loading">⏳</span> Entrando...';
                submitBtn.disabled = true;
                messageDiv.innerHTML = '';
                
                // Preparar dados do formulário
                const formData = new FormData(this);
                
                // Enviar via AJAX
                fetch('index.php', {
                    method: 'POST',
                    body: formData
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        messageDiv.innerHTML = `<div class="alert alert-success">${data.message}</div>`;
                        // Recarregar página após login bem-sucedido
                        setTimeout(() => {
                            window.location.reload();
                        }, 1000);
                    } else {
                        messageDiv.innerHTML = `<div class="alert alert-error">${data.message}</div>`;
                    }
                })
                .catch(error => {
                    messageDiv.innerHTML = `<div class="alert alert-error">Erro de conexão: ${error.message}</div>`;
                })
                .finally(() => {
                    // Restaurar botão
                    submitBtn.textContent = originalText;
                    submitBtn.disabled = false;
                });
            });
        }
    });

    // Sistema de resgate de cash via AJAX
    const claimForm = document.querySelector('.claim-form');
    if (claimForm) {
        claimForm.addEventListener('submit', function(e) {
            e.preventDefault();
            
            const submitBtn = this.querySelector('button[type="submit"]');
            const originalText = submitBtn.textContent;
            const cashValue = document.querySelector('.cash-value');
            
            // Mostrar loading
            submitBtn.innerHTML = '<span class="loading">⏳</span> Resgatando...';
            submitBtn.disabled = true;
            
            // Preparar dados
            const formData = new FormData();
            formData.append('action', 'claim_cash');
            
            // Enviar via AJAX
            fetch('index.php', {
                method: 'POST',
                body: formData
            })
            .then(response => {
                console.log('Response status:', response.status);
                console.log('Response headers:', response.headers);
                return response.text();
            })
            .then(text => {
                console.log('Raw response:', text);
                try {
                    const data = JSON.parse(text);
                    console.log('Parsed data:', data);
                    
                    if (data.success) {
                        // Atualizar cash na tela
                        if (cashValue && data.new_cash !== undefined) {
                            const newCashValue = parseInt(data.new_cash);
                            animateCashChange(newCashValue);
                        }
                        
                        // Mostrar mensagem de sucesso
                        showMessage(data.message, 'success');
                    } else {
                        showMessage(data.message, 'error');
                    }
                } catch (e) {
                    console.error('JSON parse error:', e);
                    console.log('Response was not JSON:', text);
                    showMessage('Erro: resposta inválida do servidor', 'error');
                }
            })
            .catch(error => {
                console.error('Fetch error:', error);
                showMessage('Erro de conexão: ' + error.message, 'error');
            })
            .finally(() => {
                // Restaurar botão
                submitBtn.textContent = originalText;
                submitBtn.disabled = false;
            });
        });
    }

    // Função para mostrar mensagens
    function showMessage(message, type) {
        const alertClass = type === 'success' ? 'alert-success' : 'alert-error';
        const messageDiv = document.createElement('div');
        messageDiv.className = `alert ${alertClass}`;
        messageDiv.textContent = message;
        
        // Inserir no topo do painel
        const panel = document.querySelector('.panel');
        const header = document.querySelector('.header');
        panel.insertBefore(messageDiv, header.nextSibling);
        
        // Remover após 5 segundos
        setTimeout(() => {
            messageDiv.style.transition = 'all 0.5s ease-out';
            messageDiv.style.opacity = '0';
            messageDiv.style.transform = 'translateY(-10px)';
            setTimeout(() => {
                messageDiv.remove();
            }, 500);
        }, 5000);
    }
});

// Função para criar efeito de partículas
function createParticles() {
    const container = document.querySelector('.container');
    const particleCount = 20;
    
    for (let i = 0; i < particleCount; i++) {
        createParticle();
    }
    
    function createParticle() {
        const particle = document.createElement('div');
        particle.style.position = 'absolute';
        particle.style.width = Math.random() * 4 + 1 + 'px';
        particle.style.height = particle.style.width;
        particle.style.background = 'rgba(255, 255, 255, 0.5)';
        particle.style.borderRadius = '50%';
        particle.style.pointerEvents = 'none';
        particle.style.left = Math.random() * 100 + '%';
        particle.style.top = Math.random() * 100 + '%';
        particle.style.animation = `float ${Math.random() * 10 + 10}s infinite linear`;
        
        container.appendChild(particle);
        
        // Remover partícula após animação
        setTimeout(() => {
            particle.remove();
            createParticle(); // Criar nova partícula
        }, (Math.random() * 10 + 10) * 1000);
    }
}

// Adicionar CSS para animação das partículas
const style = document.createElement('style');
style.textContent = `
    @keyframes float {
        0% {
            transform: translateY(0px) rotate(0deg);
            opacity: 1;
        }
        100% {
            transform: translateY(-100vh) rotate(360deg);
            opacity: 0;
        }
    }
    
    .loading {
        animation: spin 1s linear infinite;
    }
    
    @keyframes spin {
        0% { transform: rotate(0deg); }
        100% { transform: rotate(360deg); }
    }
`;
document.head.appendChild(style);

// Função para validar formulários
function validateForm(form) {
    const inputs = form.querySelectorAll('input[required]');
    let isValid = true;
    
    inputs.forEach(input => {
        if (!input.value.trim()) {
            input.style.borderColor = '#ff6b6b';
            input.style.boxShadow = '0 0 0 3px rgba(255, 107, 107, 0.1)';
            isValid = false;
        } else {
            input.style.borderColor = '#4facfe';
            input.style.boxShadow = '0 0 0 3px rgba(79, 172, 254, 0.1)';
        }
    });
    
    return isValid;
}

// Adicionar validação em tempo real
document.addEventListener('DOMContentLoaded', function() {
    const inputs = document.querySelectorAll('input');
    inputs.forEach(input => {
        input.addEventListener('input', function() {
            if (this.value.trim()) {
                this.style.borderColor = '#4facfe';
                this.style.boxShadow = '0 0 0 3px rgba(79, 172, 254, 0.1)';
            } else {
                this.style.borderColor = '#e1e5e9';
                this.style.boxShadow = 'none';
            }
        });
    });
});
